/*
  ============================================================================
  SCRIPT SATIŞ DÜNYASI - COMPLETE DATABASE SETUP
  ============================================================================

  Bu dosya tüm veritabanı yapısını içerir. Tek seferde çalıştırılabilir.

  İÇERİK:
  ✅ Tüm tablolar ve ilişkiler
  ✅ Sequence'ler ve fonksiyonlar
  ✅ Triggers
  ✅ RLS (Row Level Security) politikaları
  ✅ İndeksler
  ✅ Örnek veriler

  KURULUM:
  1. Supabase Dashboard > SQL Editor
  2. Bu dosyayı kopyala-yapıştır
  3. "Run" butonuna tıkla
  4. Tamamlandı!

  ============================================================================
*/

-- ============================================================================
-- 1. SEQUENCES (Sıra Numaraları)
-- ============================================================================

CREATE SEQUENCE IF NOT EXISTS order_number_seq START WITH 1001;

-- ============================================================================
-- 2. EXTENSIONS
-- ============================================================================

CREATE EXTENSION IF NOT EXISTS "uuid-ossp";
CREATE EXTENSION IF NOT EXISTS "pgcrypto";

-- ============================================================================
-- 3. CORE TABLES (Ana Tablolar)
-- ============================================================================

-- ============================================================================
-- PROFILES (Kullanıcı Profilleri - auth.users genişletmesi)
-- ============================================================================
CREATE TABLE IF NOT EXISTS profiles (
  id UUID REFERENCES auth.users(id) ON DELETE CASCADE PRIMARY KEY,
  email TEXT UNIQUE NOT NULL,
  full_name TEXT,
  avatar_url TEXT,
  role TEXT DEFAULT 'customer' CHECK (role IN ('customer', 'dealer', 'admin', 'super_admin')),
  status TEXT DEFAULT 'active' CHECK (status IN ('active', 'suspended', 'pending')),
  phone TEXT,
  address TEXT,
  company TEXT,
  dealer_package_id UUID,
  dealer_discount_rate DECIMAL(5,2) DEFAULT 0.00,
  dealer_commission_rate DECIMAL(5,2) DEFAULT 0.00,
  total_spent DECIMAL(10,2) DEFAULT 0.00,
  total_orders INTEGER DEFAULT 0,
  balance DECIMAL(10,2) DEFAULT 0, -- Cüzdan bakiyesi
  last_login TIMESTAMPTZ DEFAULT NOW(),
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- Profil oluşturma trigger'ı (auth.users'a kayıt olunca otomatik profil oluştur)
CREATE OR REPLACE FUNCTION handle_new_user()
RETURNS TRIGGER AS $$
BEGIN
  INSERT INTO public.profiles (id, email, full_name, role, created_at)
  VALUES (
    NEW.id,
    NEW.email,
    COALESCE(NEW.raw_user_meta_data->>'full_name', NEW.email),
    COALESCE(NEW.raw_user_meta_data->>'role', 'customer'),
    NOW()
  )
  ON CONFLICT (id) DO NOTHING;
  RETURN NEW;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER;

DROP TRIGGER IF EXISTS on_auth_user_created ON auth.users;
CREATE TRIGGER on_auth_user_created
  AFTER INSERT ON auth.users
  FOR EACH ROW EXECUTE FUNCTION handle_new_user();

-- ============================================================================
-- CATEGORIES (Kategoriler)
-- ============================================================================
CREATE TABLE IF NOT EXISTS categories (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  slug TEXT UNIQUE NOT NULL,
  description TEXT,
  image_url TEXT,
  parent_id UUID REFERENCES categories(id) ON DELETE SET NULL,
  sort_order INTEGER DEFAULT 0,
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- PRODUCTS (Ürünler)
-- ============================================================================
CREATE TABLE IF NOT EXISTS products (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  slug TEXT UNIQUE NOT NULL,
  description TEXT,
  short_description TEXT,
  price DECIMAL(10,2) NOT NULL DEFAULT 0,
  original_price DECIMAL(10,2),
  category_id UUID REFERENCES categories(id) ON DELETE SET NULL,
  type TEXT DEFAULT 'licensed' CHECK (type IN ('licensed', 'opensource')),
  status TEXT DEFAULT 'active' CHECK (status IN ('active', 'draft', 'inactive')),
  rating DECIMAL(3,2) DEFAULT 0.00,
  downloads INTEGER DEFAULT 0,
  stock INTEGER,
  images TEXT[] DEFAULT '{}',
  demo_url TEXT,
  download_url TEXT,
  documentation_url TEXT,
  version TEXT DEFAULT '1.0.0',
  file_size TEXT,
  requirements TEXT[] DEFAULT '{}',
  features TEXT[] DEFAULT '{}',
  tags TEXT[] DEFAULT '{}',
  is_featured BOOLEAN DEFAULT false,
  meta_title TEXT,
  meta_description TEXT,
  views INTEGER DEFAULT 0,
  sales_count INTEGER DEFAULT 0,
  last_viewed_at TIMESTAMPTZ,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- View sayısını artırma fonksiyonu
CREATE OR REPLACE FUNCTION increment_product_views(product_uuid UUID)
RETURNS VOID AS $$
BEGIN
  UPDATE products
  SET views = views + 1, last_viewed_at = NOW()
  WHERE id = product_uuid;
END;
$$ LANGUAGE plpgsql;

-- ============================================================================
-- ORDERS (Siparişler)
-- ============================================================================
CREATE TABLE IF NOT EXISTS orders (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  order_number INTEGER UNIQUE DEFAULT nextval('order_number_seq'),
  user_id UUID REFERENCES profiles(id) ON DELETE CASCADE NOT NULL,
  status TEXT DEFAULT 'pending' CHECK (status IN ('pending', 'completed', 'cancelled')),
  total_amount DECIMAL(10,2) NOT NULL DEFAULT 0,
  discount_amount DECIMAL(10,2) DEFAULT 0,
  tax_amount DECIMAL(10,2) DEFAULT 0,
  payment_method TEXT,
  payment_status TEXT DEFAULT 'pending' CHECK (payment_status IN ('pending', 'completed', 'failed', 'refunded')),
  payment_transaction_id TEXT,
  billing_name TEXT,
  billing_email TEXT,
  billing_phone TEXT,
  billing_address TEXT,
  notes TEXT,
  order_type TEXT DEFAULT 'product' CHECK (order_type IN ('product', 'dealership', 'subscription', 'renewal')),
  dealer_plan TEXT,
  license_domain TEXT,
  license_type TEXT CHECK (license_type IN ('licensed', 'opensource')),
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- ORDER_ITEMS (Sipariş Kalemleri)
-- ============================================================================
CREATE TABLE IF NOT EXISTS order_items (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  order_id UUID REFERENCES orders(id) ON DELETE CASCADE NOT NULL,
  product_id UUID REFERENCES products(id) ON DELETE CASCADE NOT NULL,
  product_name TEXT NOT NULL,
  product_price DECIMAL(10,2) NOT NULL,
  quantity INTEGER DEFAULT 1,
  total_price DECIMAL(10,2) NOT NULL,
  download_url TEXT,
  max_downloads INTEGER DEFAULT 5,
  download_count INTEGER DEFAULT 0,
  download_expires_at TIMESTAMPTZ DEFAULT (NOW() + INTERVAL '1 year'),
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- 4. DEALER SYSTEM (Bayilik Sistemi)
-- ============================================================================

-- ============================================================================
-- DEALER_PACKAGES (Bayilik Paketleri)
-- ============================================================================
CREATE TABLE IF NOT EXISTS dealer_packages (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  slug TEXT UNIQUE NOT NULL,
  description TEXT,
  price DECIMAL(10,2) NOT NULL DEFAULT 0,
  billing_period TEXT DEFAULT '1_year' CHECK (billing_period IN ('1_month', '3_months', '6_months', '1_year')),
  duration_months INTEGER DEFAULT 12,
  display_name TEXT DEFAULT 'Yıllık',
  features TEXT[] DEFAULT '{}',
  discount_rate DECIMAL(5,2) DEFAULT 0.00,
  commission_rate DECIMAL(5,2) DEFAULT 0.00,
  max_products INTEGER DEFAULT 100,
  max_users INTEGER DEFAULT 0,
  priority_support BOOLEAN DEFAULT false,
  is_active BOOLEAN DEFAULT true,
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- DEALER_APPLICATIONS (Bayilik Başvuruları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS dealer_applications (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id) ON DELETE CASCADE NOT NULL,
  package_id UUID REFERENCES dealer_packages(id) ON DELETE CASCADE NOT NULL,
  company_name TEXT NOT NULL,
  tax_number TEXT,
  phone TEXT NOT NULL,
  address TEXT,
  website TEXT,
  message TEXT,
  status TEXT DEFAULT 'pending' CHECK (status IN ('pending', 'approved', 'rejected')),
  admin_notes TEXT,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- DEALERS (Bayiler)
-- ============================================================================
CREATE TABLE IF NOT EXISTS dealers (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES profiles(id) ON DELETE CASCADE UNIQUE,
  company_name TEXT,
  name TEXT,
  email TEXT,
  phone TEXT,
  plan TEXT,
  plan_id TEXT NOT NULL,
  plan_name TEXT,
  plan_type TEXT CHECK (plan_type IN ('bronze', 'silver', 'gold')),
  status TEXT DEFAULT 'active',
  discount_rate INTEGER DEFAULT 0,
  commission DECIMAL(10,2) DEFAULT 0,
  total_sales DECIMAL(10,2) DEFAULT 0,

  -- Paket bilgileri
  price DECIMAL(10,2),
  package_price DECIMAL(10,2) DEFAULT 0,
  billing_period TEXT,
  billing_period_label TEXT,
  duration_months INTEGER,
  monthly_price DECIMAL(10,2),
  monthly_equivalent DECIMAL(10,2),
  period_discount DECIMAL(10,2),
  period_savings DECIMAL(10,2),

  -- Tarihler
  start_date TIMESTAMPTZ,
  end_date TIMESTAMPTZ,
  expiry_date TIMESTAMPTZ,
  purchase_date TIMESTAMPTZ DEFAULT NOW(),
  next_payment_date TIMESTAMPTZ,

  -- Limitler
  max_downloads INTEGER DEFAULT 0,
  current_downloads INTEGER DEFAULT 0,
  download_limit INTEGER,
  download_used INTEGER DEFAULT 0,
  license_limit INTEGER DEFAULT 10,
  license_used INTEGER DEFAULT 0,
  last_license_reset TIMESTAMPTZ DEFAULT date_trunc('month', CURRENT_TIMESTAMP),

  -- Özellikler
  features JSONB,
  support_level TEXT DEFAULT 'standard',

  -- API ve Özelleştirme
  api_key TEXT UNIQUE,
  api_enabled BOOLEAN DEFAULT false,
  theme_preference TEXT DEFAULT 'light' CHECK (theme_preference IN ('light', 'dark')),
  custom_logo_url TEXT,
  branding_enabled BOOLEAN DEFAULT false,
  white_label_domain TEXT,

  -- Referans sistemi
  referral_code TEXT UNIQUE,
  referral_earnings DECIMAL(10,2) DEFAULT 0,

  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- DEALER_CUSTOMERS (Bayi Müşterileri)
-- ============================================================================
CREATE TABLE IF NOT EXISTS dealer_customers (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  dealer_id UUID REFERENCES dealers(id) ON DELETE CASCADE NOT NULL,
  customer_name TEXT NOT NULL,
  customer_email TEXT NOT NULL,
  customer_phone TEXT,
  total_purchases DECIMAL(10,2) DEFAULT 0,
  last_purchase_date TIMESTAMPTZ,
  notes TEXT,
  tags TEXT[],
  is_active BOOLEAN DEFAULT true,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- DEALER_LICENSES (Bayi Lisansları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS dealer_licenses (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  dealer_id UUID REFERENCES profiles(id) ON DELETE CASCADE NOT NULL,
  customer_id UUID REFERENCES dealer_customers(id) ON DELETE SET NULL,
  license_key TEXT UNIQUE NOT NULL,
  product_name TEXT NOT NULL,
  customer_name TEXT NOT NULL,
  customer_email TEXT NOT NULL,
  domain TEXT,
  status TEXT DEFAULT 'active' CHECK (status IN ('active', 'expired', 'suspended')),
  max_activations INTEGER DEFAULT 1,
  current_activations INTEGER DEFAULT 0,
  expires_at TIMESTAMPTZ NOT NULL,
  notes TEXT,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- LICENSE_ACTIVATIONS (Lisans Aktivasyonları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS license_activations (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  license_id UUID REFERENCES dealer_licenses(id) ON DELETE CASCADE NOT NULL,
  domain TEXT NOT NULL,
  ip_address TEXT,
  user_agent TEXT,
  activated_at TIMESTAMPTZ DEFAULT NOW(),
  last_seen_at TIMESTAMPTZ DEFAULT NOW(),
  activation_count INTEGER DEFAULT 1
);

-- ============================================================================
-- LICENSE_NOTES (Lisans Notları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS license_notes (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  license_id UUID REFERENCES dealer_licenses(id) ON DELETE CASCADE,
  dealer_id UUID REFERENCES dealers(id) ON DELETE CASCADE,
  note TEXT NOT NULL,
  created_by UUID REFERENCES auth.users(id) ON DELETE SET NULL,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- DEALER_SALES (Bayi Satışları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS dealer_sales (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  dealer_id UUID REFERENCES dealers(id) ON DELETE CASCADE NOT NULL,
  order_id UUID REFERENCES orders(id) ON DELETE SET NULL,
  customer_name TEXT NOT NULL,
  customer_email TEXT NOT NULL,
  product_name TEXT NOT NULL,
  sale_amount DECIMAL(10,2) NOT NULL,
  dealer_commission DECIMAL(10,2) DEFAULT 0,
  discount_applied DECIMAL(10,2) DEFAULT 0,
  sale_date TIMESTAMPTZ DEFAULT NOW(),
  status TEXT DEFAULT 'completed',
  payment_method TEXT,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- DEALER_DOWNLOADS (Bayi İndirmeleri)
-- ============================================================================
CREATE TABLE IF NOT EXISTS dealer_downloads (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  dealer_id UUID REFERENCES dealers(id) ON DELETE CASCADE NOT NULL,
  product_id UUID REFERENCES products(id) ON DELETE SET NULL,
  product_name TEXT NOT NULL,
  download_date TIMESTAMPTZ DEFAULT NOW(),
  file_size BIGINT DEFAULT 0,
  download_count INTEGER DEFAULT 1,
  ip_address TEXT,
  user_agent TEXT,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- DEALER_SUPPORT_TICKETS (Bayi Destek Talepleri)
-- ============================================================================
CREATE TABLE IF NOT EXISTS dealer_support_tickets (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  dealer_id UUID REFERENCES dealers(id) ON DELETE CASCADE NOT NULL,
  ticket_number TEXT UNIQUE NOT NULL,
  subject TEXT NOT NULL,
  description TEXT NOT NULL,
  priority TEXT DEFAULT 'normal' CHECK (priority IN ('low', 'normal', 'high', 'urgent')),
  status TEXT DEFAULT 'open' CHECK (status IN ('open', 'in_progress', 'resolved', 'closed')),
  support_level TEXT DEFAULT 'email',
  assigned_to TEXT,
  response_time_hours INTEGER,
  resolved_at TIMESTAMPTZ,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- DEALER_API_USAGE (Bayi API Kullanımı)
-- ============================================================================
CREATE TABLE IF NOT EXISTS dealer_api_usage (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  dealer_id UUID REFERENCES dealers(id) ON DELETE CASCADE NOT NULL,
  endpoint TEXT NOT NULL,
  method TEXT NOT NULL,
  request_count INTEGER DEFAULT 1,
  response_time_ms INTEGER,
  status_code INTEGER,
  ip_address TEXT,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- 5. CONTENT TABLES (İçerik Tabloları)
-- ============================================================================

-- ============================================================================
-- BLOG_POSTS (Blog Yazıları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS blog_posts (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  title TEXT NOT NULL,
  slug TEXT UNIQUE NOT NULL,
  excerpt TEXT,
  content TEXT,
  featured_image TEXT,
  author_id UUID REFERENCES profiles(id) ON DELETE CASCADE NOT NULL,
  author_name TEXT,
  category TEXT,
  tags TEXT[] DEFAULT '{}',
  status TEXT DEFAULT 'draft' CHECK (status IN ('published', 'draft', 'scheduled')),
  is_featured BOOLEAN DEFAULT false,
  view_count INTEGER DEFAULT 0,
  comment_count INTEGER DEFAULT 0,
  meta_title TEXT,
  meta_description TEXT,
  published_at TIMESTAMPTZ,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- PRODUCT_REVIEWS (Ürün Yorumları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS product_reviews (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  product_id UUID REFERENCES products(id) ON DELETE CASCADE NOT NULL,
  user_id UUID REFERENCES profiles(id) ON DELETE SET NULL,
  user_name TEXT NOT NULL,
  user_email TEXT NOT NULL,
  rating INTEGER NOT NULL CHECK (rating >= 1 AND rating <= 5),
  title TEXT,
  comment TEXT NOT NULL,
  is_approved BOOLEAN DEFAULT false,
  is_verified_purchase BOOLEAN DEFAULT false,
  helpful_count INTEGER DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- PRODUCT_VIEWS (Ürün Görüntüleme Analitikleri)
-- ============================================================================
CREATE TABLE IF NOT EXISTS product_views (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  product_id UUID REFERENCES products(id) ON DELETE CASCADE,
  viewed_at TIMESTAMPTZ DEFAULT NOW(),
  session_id TEXT,
  user_agent TEXT,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- SLIDERS (Slider/Kaydırıcılar)
-- ============================================================================
CREATE TABLE IF NOT EXISTS sliders (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  title TEXT NOT NULL,
  subtitle TEXT,
  description TEXT,
  image_url TEXT NOT NULL,
  link_url TEXT,
  button_text TEXT DEFAULT 'Detaylı Bilgi',
  status TEXT DEFAULT 'active' CHECK (status IN ('active', 'inactive')),
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- BANNERS (Banner Reklamlar)
-- ============================================================================
CREATE TABLE IF NOT EXISTS banners (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  title TEXT NOT NULL,
  description TEXT,
  image_url TEXT NOT NULL,
  link_url TEXT,
  button_text TEXT,
  position TEXT DEFAULT 'home_top',
  status TEXT DEFAULT 'active' CHECK (status IN ('active', 'inactive')),
  sort_order INTEGER DEFAULT 0,
  start_date TIMESTAMPTZ,
  end_date TIMESTAMPTZ,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- SERVICES (Hizmetler)
-- ============================================================================
CREATE TABLE IF NOT EXISTS services (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  title TEXT NOT NULL,
  description TEXT NOT NULL,
  icon TEXT DEFAULT 'Box',
  image_url TEXT,
  link_url TEXT,
  status TEXT DEFAULT 'active' CHECK (status IN ('active', 'inactive')),
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- PAGES (Sayfalar)
-- ============================================================================
CREATE TABLE IF NOT EXISTS pages (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  title TEXT NOT NULL,
  slug TEXT UNIQUE NOT NULL,
  content TEXT NOT NULL,
  meta_description TEXT,
  meta_keywords TEXT,
  image_url TEXT,
  status TEXT DEFAULT 'published' CHECK (status IN ('published', 'draft', 'archived')),
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- TESTIMONIALS (Müşteri Yorumları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS testimonials (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  customer_name TEXT NOT NULL,
  customer_company TEXT,
  customer_role TEXT,
  customer_image TEXT,
  rating INTEGER DEFAULT 5 CHECK (rating >= 1 AND rating <= 5),
  comment TEXT NOT NULL,
  status TEXT DEFAULT 'approved' CHECK (status IN ('approved', 'pending', 'rejected')),
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- INTEGRATIONS (Entegrasyonlar)
-- ============================================================================
CREATE TABLE IF NOT EXISTS integrations (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  category TEXT NOT NULL,
  description TEXT NOT NULL,
  features JSONB DEFAULT '[]',
  price DECIMAL(10,2) DEFAULT 0,
  popular BOOLEAN DEFAULT false,
  badge TEXT,
  color TEXT DEFAULT 'from-blue-500 to-cyan-600',
  icon_name TEXT DEFAULT 'Package',
  status TEXT DEFAULT 'active' CHECK (status IN ('active', 'inactive')),
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- WEB_DESIGNS (Web Tasarımları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS web_designs (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  description TEXT NOT NULL,
  price TEXT NOT NULL,
  features JSONB DEFAULT '[]',
  color TEXT DEFAULT 'from-blue-500 to-cyan-600',
  icon_name TEXT DEFAULT 'Monitor',
  popular BOOLEAN DEFAULT false,
  status TEXT DEFAULT 'active' CHECK (status IN ('active', 'inactive')),
  sort_order INTEGER DEFAULT 0,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- 6. SYSTEM TABLES (Sistem Tabloları)
-- ============================================================================

-- ============================================================================
-- SITE_SETTINGS (Site Ayarları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS site_settings (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  site_name TEXT DEFAULT 'Script Satış Dünyası',
  site_description TEXT,
  site_url TEXT,
  site_logo TEXT,
  favicon TEXT,

  -- İletişim
  admin_email TEXT,
  contact_email TEXT,
  contact_phone TEXT,
  whatsapp_number TEXT,
  address TEXT,

  -- Sosyal Medya
  social_facebook TEXT,
  social_twitter TEXT,
  social_instagram TEXT,
  social_linkedin TEXT,

  -- Hero Section
  hero_title TEXT,
  hero_subtitle TEXT,
  hero_description TEXT,
  hero_button_text TEXT,
  hero_button_link TEXT,
  hero_second_button_text TEXT,
  hero_second_button_link TEXT,
  hero_background_image TEXT,

  -- İstatistikler
  stats_customers INTEGER DEFAULT 0,
  stats_scripts INTEGER DEFAULT 0,
  stats_downloads INTEGER DEFAULT 0,
  stats_rating DECIMAL(2,1) DEFAULT 5.0,

  -- SEO
  meta_title TEXT,
  meta_description TEXT,
  meta_keywords TEXT,
  google_analytics TEXT,
  auto_sitemap BOOLEAN DEFAULT false,
  use_robots BOOLEAN DEFAULT true,
  schema_org BOOLEAN DEFAULT true,

  -- Ödeme (PayTR)
  paytr_enabled BOOLEAN DEFAULT false,
  paytr_merchant_id TEXT,
  paytr_merchant_key TEXT,
  paytr_merchant_salt TEXT,
  paytr_test_mode BOOLEAN DEFAULT true,

  -- EFT/Havale
  eft_bank_name TEXT DEFAULT 'Ziraat Bankası',
  eft_account_holder TEXT DEFAULT 'SCRIPT SATIŞ DÜNYASI',
  eft_iban TEXT DEFAULT 'TR00 0000 0000 0000 0000 0000 00',
  eft_instructions TEXT DEFAULT 'Lütfen havale/EFT açıklama kısmına referans kodunu yazınız.',

  -- Vergi
  tax_rate DECIMAL(5,2) DEFAULT 0,
  tax_enabled BOOLEAN DEFAULT false,
  tax_number TEXT,
  tax_office TEXT,
  company_name TEXT,
  show_vat_breakdown BOOLEAN DEFAULT false,
  require_tax_info BOOLEAN DEFAULT false,
  auto_invoice BOOLEAN DEFAULT false,

  -- Email (SMTP)
  smtp_host TEXT,
  smtp_port INTEGER,
  smtp_username TEXT,
  smtp_password TEXT,
  smtp_encryption TEXT,
  sender_name TEXT,
  sender_email TEXT,

  -- Harita
  map_iframe TEXT,
  map_coordinates TEXT,
  map_zoom INTEGER DEFAULT 15,
  show_map BOOLEAN DEFAULT true,

  -- Diğer
  maintenance_mode BOOLEAN DEFAULT false,
  custom_css TEXT,
  custom_js TEXT,
  footer_text TEXT,
  footer_links TEXT,

  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- ADMIN_USERS (Admin Kullanıcıları)
-- ============================================================================
CREATE TABLE IF NOT EXISTS admin_users (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  email TEXT UNIQUE NOT NULL,
  name TEXT NOT NULL,
  password_hash TEXT NOT NULL,
  role TEXT DEFAULT 'admin' CHECK (role IN ('super_admin', 'admin')),
  permissions JSONB DEFAULT '{}',
  is_active BOOLEAN DEFAULT true,
  last_login TIMESTAMPTZ,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  created_by UUID,
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- LICENSES (Lisanslar)
-- ============================================================================
CREATE TABLE IF NOT EXISTS licenses (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  license_key TEXT UNIQUE NOT NULL,
  license_type TEXT NOT NULL CHECK (license_type IN ('DEMO', 'SINGLE_USE', 'SOURCE_CODE')),
  customer_name TEXT NOT NULL,
  customer_email TEXT NOT NULL,
  customer_phone TEXT,
  licensed_domain TEXT NOT NULL,
  issued_at TIMESTAMPTZ DEFAULT NOW(),
  expires_at TIMESTAMPTZ,
  support_ends_at TIMESTAMPTZ NOT NULL,
  is_active BOOLEAN DEFAULT true,
  notes TEXT,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- NEWSLETTER_SUBSCRIPTIONS (Newsletter Abonelikleri)
-- ============================================================================
CREATE TABLE IF NOT EXISTS newsletter_subscriptions (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  email TEXT UNIQUE NOT NULL,
  name TEXT,
  status TEXT DEFAULT 'active' CHECK (status IN ('active', 'unsubscribed', 'bounced')),
  tags TEXT[] DEFAULT '{}',
  metadata JSONB DEFAULT '{}',
  subscribed_at TIMESTAMPTZ DEFAULT NOW(),
  unsubscribed_at TIMESTAMPTZ,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- CART_ITEMS (Sepet Kalemleri)
-- ============================================================================
CREATE TABLE IF NOT EXISTS cart_items (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id) ON DELETE CASCADE,
  session_id TEXT,
  product_id UUID REFERENCES products(id) ON DELETE CASCADE NOT NULL,
  quantity INTEGER DEFAULT 1 CHECK (quantity > 0),
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- WALLET_TRANSACTIONS (Cüzdan İşlemleri)
-- ============================================================================
CREATE TABLE IF NOT EXISTS wallet_transactions (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES profiles(id) ON DELETE CASCADE NOT NULL,
  amount DECIMAL(10,2) NOT NULL,
  type TEXT NOT NULL CHECK (type IN ('deposit', 'payment', 'refund')),
  description TEXT NOT NULL,
  order_id UUID REFERENCES orders(id) ON DELETE SET NULL,
  balance_after DECIMAL(10,2) NOT NULL,
  status TEXT DEFAULT 'completed' CHECK (status IN ('pending', 'completed', 'failed')),
  payment_method TEXT,
  transaction_id TEXT UNIQUE,
  created_at TIMESTAMPTZ DEFAULT NOW()
);

-- ============================================================================
-- 7. INDEXES (İndeksler - Performans için)
-- ============================================================================

CREATE INDEX IF NOT EXISTS idx_products_category ON products(category_id);
CREATE INDEX IF NOT EXISTS idx_products_slug ON products(slug);
CREATE INDEX IF NOT EXISTS idx_products_status ON products(status);
CREATE INDEX IF NOT EXISTS idx_products_featured ON products(is_featured);

CREATE INDEX IF NOT EXISTS idx_orders_user ON orders(user_id);
CREATE INDEX IF NOT EXISTS idx_orders_status ON orders(status);
CREATE INDEX IF NOT EXISTS idx_orders_created ON orders(created_at DESC);

CREATE INDEX IF NOT EXISTS idx_dealer_licenses_dealer ON dealer_licenses(dealer_id);
CREATE INDEX IF NOT EXISTS idx_dealer_licenses_status ON dealer_licenses(status);

CREATE INDEX IF NOT EXISTS idx_blog_posts_status ON blog_posts(status);
CREATE INDEX IF NOT EXISTS idx_blog_posts_slug ON blog_posts(slug);

-- ============================================================================
-- 8. RLS (Row Level Security) POLİTİKALARI
-- ============================================================================

-- Tüm tablolarda RLS etkinleştir
ALTER TABLE profiles ENABLE ROW LEVEL SECURITY;
ALTER TABLE categories ENABLE ROW LEVEL SECURITY;
ALTER TABLE products ENABLE ROW LEVEL SECURITY;
ALTER TABLE orders ENABLE ROW LEVEL SECURITY;
ALTER TABLE order_items ENABLE ROW LEVEL SECURITY;
ALTER TABLE dealer_packages ENABLE ROW LEVEL SECURITY;
ALTER TABLE dealer_applications ENABLE ROW LEVEL SECURITY;
ALTER TABLE dealers ENABLE ROW LEVEL SECURITY;
ALTER TABLE dealer_customers ENABLE ROW LEVEL SECURITY;
ALTER TABLE dealer_licenses ENABLE ROW LEVEL SECURITY;
ALTER TABLE license_activations ENABLE ROW LEVEL SECURITY;
ALTER TABLE license_notes ENABLE ROW LEVEL SECURITY;
ALTER TABLE dealer_sales ENABLE ROW LEVEL SECURITY;
ALTER TABLE dealer_downloads ENABLE ROW LEVEL SECURITY;
ALTER TABLE dealer_support_tickets ENABLE ROW LEVEL SECURITY;
ALTER TABLE dealer_api_usage ENABLE ROW LEVEL SECURITY;
ALTER TABLE blog_posts ENABLE ROW LEVEL SECURITY;
ALTER TABLE product_reviews ENABLE ROW LEVEL SECURITY;
ALTER TABLE product_views ENABLE ROW LEVEL SECURITY;
ALTER TABLE sliders ENABLE ROW LEVEL SECURITY;
ALTER TABLE services ENABLE ROW LEVEL SECURITY;
ALTER TABLE pages ENABLE ROW LEVEL SECURITY;
ALTER TABLE testimonials ENABLE ROW LEVEL SECURITY;
ALTER TABLE integrations ENABLE ROW LEVEL SECURITY;
ALTER TABLE web_designs ENABLE ROW LEVEL SECURITY;
ALTER TABLE site_settings ENABLE ROW LEVEL SECURITY;
ALTER TABLE licenses ENABLE ROW LEVEL SECURITY;
ALTER TABLE newsletter_subscriptions ENABLE ROW LEVEL SECURITY;
ALTER TABLE cart_items ENABLE ROW LEVEL SECURITY;
ALTER TABLE wallet_transactions ENABLE ROW LEVEL SECURITY;

-- ============================================================================
-- PROFILES POLICIES
-- ============================================================================
DROP POLICY IF EXISTS "Users can view own profile" ON profiles;
CREATE POLICY "Users can view own profile" ON profiles
  FOR SELECT TO authenticated
  USING (auth.uid() = id);

DROP POLICY IF EXISTS "Users can update own profile" ON profiles;
CREATE POLICY "Users can update own profile" ON profiles
  FOR UPDATE TO authenticated
  USING (auth.uid() = id)
  WITH CHECK (auth.uid() = id);

DROP POLICY IF EXISTS "Super admins can view all profiles" ON profiles;
CREATE POLICY "Super admins can view all profiles" ON profiles
  FOR SELECT TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles p
      WHERE p.id = auth.uid() AND p.role = 'super_admin'
    )
  );

DROP POLICY IF EXISTS "Super admins can update all profiles" ON profiles;
CREATE POLICY "Super admins can update all profiles" ON profiles
  FOR UPDATE TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles p
      WHERE p.id = auth.uid() AND p.role = 'super_admin'
    )
  );

-- ============================================================================
-- CATEGORIES POLICIES
-- ============================================================================
DROP POLICY IF EXISTS "Anyone can view active categories" ON categories;
CREATE POLICY "Anyone can view active categories" ON categories
  FOR SELECT TO anon, authenticated
  USING (is_active = true);

DROP POLICY IF EXISTS "Super admins can manage categories" ON categories;
CREATE POLICY "Super admins can manage categories" ON categories
  FOR ALL TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles
      WHERE id = auth.uid() AND role = 'super_admin'
    )
  );

-- ============================================================================
-- PRODUCTS POLICIES
-- ============================================================================
DROP POLICY IF EXISTS "Anyone can view active products" ON products;
CREATE POLICY "Anyone can view active products" ON products
  FOR SELECT TO anon, authenticated
  USING (status = 'active');

DROP POLICY IF EXISTS "Super admins can manage products" ON products;
CREATE POLICY "Super admins can manage products" ON products
  FOR ALL TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles
      WHERE id = auth.uid() AND role = 'super_admin'
    )
  );

-- ============================================================================
-- ORDERS POLICIES
-- ============================================================================
DROP POLICY IF EXISTS "Users can view own orders" ON orders;
CREATE POLICY "Users can view own orders" ON orders
  FOR SELECT TO authenticated
  USING (user_id = auth.uid());

DROP POLICY IF EXISTS "Users can create orders" ON orders;
CREATE POLICY "Users can create orders" ON orders
  FOR INSERT TO authenticated
  WITH CHECK (user_id = auth.uid());

DROP POLICY IF EXISTS "Super admins can view all orders" ON orders;
CREATE POLICY "Super admins can view all orders" ON orders
  FOR SELECT TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles
      WHERE id = auth.uid() AND role = 'super_admin'
    )
  );

DROP POLICY IF EXISTS "Super admins can manage orders" ON orders;
CREATE POLICY "Super admins can manage orders" ON orders
  FOR ALL TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles
      WHERE id = auth.uid() AND role = 'super_admin'
    )
  );

-- ============================================================================
-- ORDER_ITEMS POLICIES
-- ============================================================================
DROP POLICY IF EXISTS "Users can view own order items" ON order_items;
CREATE POLICY "Users can view own order items" ON order_items
  FOR SELECT TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM orders
      WHERE orders.id = order_items.order_id
      AND orders.user_id = auth.uid()
    )
  );

DROP POLICY IF EXISTS "Super admins can manage order items" ON order_items;
CREATE POLICY "Super admins can manage order items" ON order_items
  FOR ALL TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles
      WHERE id = auth.uid() AND role = 'super_admin'
    )
  );

-- ============================================================================
-- DEALER POLICIES
-- ============================================================================
DROP POLICY IF EXISTS "Dealers can view own data" ON dealers;
CREATE POLICY "Dealers can view own data" ON dealers
  FOR SELECT TO authenticated
  USING (user_id = auth.uid());

DROP POLICY IF EXISTS "Dealers can update own data" ON dealers;
CREATE POLICY "Dealers can update own data" ON dealers
  FOR UPDATE TO authenticated
  USING (user_id = auth.uid())
  WITH CHECK (user_id = auth.uid());

DROP POLICY IF EXISTS "Super admins can manage all dealers" ON dealers;
CREATE POLICY "Super admins can manage all dealers" ON dealers
  FOR ALL TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles
      WHERE id = auth.uid() AND role = 'super_admin'
    )
  );

-- ============================================================================
-- DEALER_LICENSES POLICIES
-- ============================================================================
DROP POLICY IF EXISTS "Dealers can view own licenses" ON dealer_licenses;
CREATE POLICY "Dealers can view own licenses" ON dealer_licenses
  FOR SELECT TO authenticated
  USING (dealer_id = auth.uid());

DROP POLICY IF EXISTS "Dealers can create licenses" ON dealer_licenses;
CREATE POLICY "Dealers can create licenses" ON dealer_licenses
  FOR INSERT TO authenticated
  WITH CHECK (dealer_id = auth.uid());

DROP POLICY IF EXISTS "Dealers can update own licenses" ON dealer_licenses;
CREATE POLICY "Dealers can update own licenses" ON dealer_licenses
  FOR UPDATE TO authenticated
  USING (dealer_id = auth.uid())
  WITH CHECK (dealer_id = auth.uid());

DROP POLICY IF EXISTS "Super admins can manage all licenses" ON dealer_licenses;
CREATE POLICY "Super admins can manage all licenses" ON dealer_licenses
  FOR ALL TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles
      WHERE id = auth.uid() AND role = 'super_admin'
    )
  );

-- ============================================================================
-- SITE_SETTINGS POLICIES
-- ============================================================================
DROP POLICY IF EXISTS "Anyone can view site settings" ON site_settings;
CREATE POLICY "Anyone can view site settings" ON site_settings
  FOR SELECT TO anon, authenticated
  USING (true);

DROP POLICY IF EXISTS "Super admins can manage site settings" ON site_settings;
CREATE POLICY "Super admins can manage site settings" ON site_settings
  FOR ALL TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles
      WHERE id = auth.uid() AND role = 'super_admin'
    )
  );

-- ============================================================================
-- CART_ITEMS POLICIES
-- ============================================================================
DROP POLICY IF EXISTS "Users can view own cart" ON cart_items;
CREATE POLICY "Users can view own cart" ON cart_items
  FOR SELECT TO authenticated
  USING (user_id = auth.uid());

DROP POLICY IF EXISTS "Users can manage own cart" ON cart_items;
CREATE POLICY "Users can manage own cart" ON cart_items
  FOR ALL TO authenticated
  USING (user_id = auth.uid())
  WITH CHECK (user_id = auth.uid());

-- ============================================================================
-- WALLET_TRANSACTIONS POLICIES
-- ============================================================================
DROP POLICY IF EXISTS "Users can view own transactions" ON wallet_transactions;
CREATE POLICY "Users can view own transactions" ON wallet_transactions
  FOR SELECT TO authenticated
  USING (user_id = auth.uid());

DROP POLICY IF EXISTS "Super admins can manage transactions" ON wallet_transactions;
CREATE POLICY "Super admins can manage transactions" ON wallet_transactions
  FOR ALL TO authenticated
  USING (
    EXISTS (
      SELECT 1 FROM profiles
      WHERE id = auth.uid() AND role = 'super_admin'
    )
  );

-- ============================================================================
-- PUBLIC READ POLICIES (Herkes okuyabilir)
-- ============================================================================
DROP POLICY IF EXISTS "Anyone can view active blog posts" ON blog_posts;
CREATE POLICY "Anyone can view active blog posts" ON blog_posts
  FOR SELECT TO anon, authenticated
  USING (status = 'published');

DROP POLICY IF EXISTS "Anyone can view approved reviews" ON product_reviews;
CREATE POLICY "Anyone can view approved reviews" ON product_reviews
  FOR SELECT TO anon, authenticated
  USING (is_approved = true);

DROP POLICY IF EXISTS "Anyone can view active sliders" ON sliders;
CREATE POLICY "Anyone can view active sliders" ON sliders
  FOR SELECT TO anon, authenticated
  USING (status = 'active');

DROP POLICY IF EXISTS "Anyone can view active services" ON services;
CREATE POLICY "Anyone can view active services" ON services
  FOR SELECT TO anon, authenticated
  USING (status = 'active');

DROP POLICY IF EXISTS "Anyone can view published pages" ON pages;
CREATE POLICY "Anyone can view published pages" ON pages
  FOR SELECT TO anon, authenticated
  USING (status = 'published');

DROP POLICY IF EXISTS "Anyone can view approved testimonials" ON testimonials;
CREATE POLICY "Anyone can view approved testimonials" ON testimonials
  FOR SELECT TO anon, authenticated
  USING (status = 'approved');

DROP POLICY IF EXISTS "Anyone can view dealer packages" ON dealer_packages;
CREATE POLICY "Anyone can view dealer packages" ON dealer_packages
  FOR SELECT TO anon, authenticated
  USING (is_active = true);

DROP POLICY IF EXISTS "Anyone can view integrations" ON integrations;
CREATE POLICY "Anyone can view integrations" ON integrations
  FOR SELECT TO anon, authenticated
  USING (status = 'active');

DROP POLICY IF EXISTS "Anyone can view web designs" ON web_designs;
CREATE POLICY "Anyone can view web designs" ON web_designs
  FOR SELECT TO anon, authenticated
  USING (status = 'active');

-- ============================================================================
-- 9. INITIAL DATA (Başlangıç Verileri)
-- ============================================================================

-- Site ayarlarını oluştur
INSERT INTO site_settings (
  site_name,
  site_description,
  contact_email,
  contact_phone,
  address,
  stats_customers,
  stats_scripts,
  stats_downloads,
  stats_rating
) VALUES (
  'Script Satış Dünyası',
  'Profesyonel web scriptleri ve yazılım çözümleri',
  'info@scriptsatisdunyasi.com.tr',
  '+90 (552) 065 71 21',
  'İstanbul, Türkiye',
  5000,
  150,
  25000,
  4.8
) ON CONFLICT DO NOTHING;

-- ============================================================================
-- SON MESAJ
-- ============================================================================
DO $$
BEGIN
  RAISE NOTICE '
  ============================================================================
  ✅ KURULUM TAMAMLANDI!
  ============================================================================

  Tüm tablolar, ilişkiler, RLS politikaları ve indeksler başarıyla oluşturuldu.

  SONRAKI ADIMLAR:
  1. Admin kullanıcısı oluştur
  2. Kategorileri ekle
  3. Ürünleri yükle
  4. Site ayarlarını düzenle

  ============================================================================
  ';
END $$;
